import { create } from "zustand";
import { devtools, persist } from "zustand/middleware";
import type { User } from "../types";
import { users } from "../utils/users";
import createSelectors from "./create-selector";

type UsersStore = {
  users: User[];
  getUserById: (id: string) => User | undefined;
  addUsers: (user: User) => void;
  updateUser: (id: string, updatedUser: User) => void;
  deleteUser: (id: string) => void;
};

const useUsersStore = create<UsersStore>()(
  persist(
    devtools((set, get) => ({
      users: users,
      getUserById: (id) => {
        return get().users.find((user) => user.id === id);
      },
      addUsers: (user) => {
        set({
          users: [user, ...get().users],
        });
      },
      updateUser: (id, updatedUser) => {
        set({
          users: get().users.map((user) => {
            if (user.id === id) {
              return {
                ...user,
                ...updatedUser,
                id,
              };
            }
            return user;
          }),
        });
      },
      deleteUser: (id) => {
        set({
          users: get().users.filter((user) => user.id !== id),
        });
      },
    })),
    {
      name: "users",
    }
  )
);

export const usersSelector = createSelectors(useUsersStore);

export default useUsersStore;
